classdef ductedfan < handle
    
    properties
        param = struct('m', 6,...
                       'd', 0.1,...
                       'g', 9.8,...
                       'r', 0.25,...
                       'J', 0.1425)
        
        tSim
        xSim            % states are x, y, theta, dx, dy, dtheta
        xdotSim
        uSim            % controls are tau and psi
        
        % Most likely useful for other implementations, possibly as
        % subclss. Think through how coding design permits the different
        % versions to easily run.
        %A
        %B
        %P
        %Klqr
    end
    
    methods
        %========================== ductedfan ==========================
        %
        % Constructor for the inverted pendulum on a cart instance.
        %
        function obj = ductedfan(param)
             if exist('param','var')
                  obj.param = param;
             end
        end
        
        %============================ runSim ===========================
        %
        % Abstracted interface to run the simulation of the ducted
        % fan. Uses a format similar to odeXX functions for specifying the
        % initial value problem (IVP) with the addition of a control
        % function.
        %
        function [tSim, xSim, xdotSim, uSim] = runSim(self, tspan, x0, u)
            sys = @(t,x) self.dynamics(t, x, u);

            % Simulate the system.
            [tSim, xSim] = ode45(sys, tspan, x0);

            % Use simulation data to recover controls and diff eq.
            % Data is row-wise.
            num     = length(tSim);
            uSim    = zeros(num,2);
            xdotSim = zeros(size(xSim));
            
            for i = 1:num
                x = xSim(i,:)';
                t = tSim(i);
                uSim(i,:) = u(t, x)';
                xdotSim(i,:) = self.dynamics(t, x, u)';
            end
            
            % Store variables in case needed for later.
            self.tSim = tSim;
            self.xSim = xSim;
            self.xdotSim = xdotSim;
            self.uSim = uSim;
        end
        
        %=========================== dynamics ==========================
        %
        % Compute the dynamics of the system under the applied control.
        %
        function xdot = dynamics(self, t, xvec, uvec)
            x  = xvec(1); y  = xvec(2); th  = xvec(3);
            dx = xvec(4); dy = xvec(5); dth = xvec(6);

            q = [x;y];
            dq = [dx;dy];
            
            u = uvec(t,xvec);
            tau = u(1);
            psi = u(2);
            
            m = self.param.m;
            d = self.param.d;
            g = self.param.g;
            r = self.param.r;
            J = self.param.J;

            Rth  = self.rotationR(th);
            Rpsi = self.rotationR(-psi);
            
            qdot  = dq;
            qddot = -d/m*dq + 1/m*Rth*Rpsi*[0;tau] - [0;g];
            
            thdot  = dth;
            thddot = -r/J*sin(psi)*tau;
            
            xdot   = [qdot; thdot; qddot; thddot];
            
        end

        %=========================== rotation ==========================
        %
        % Helper function. Not best place to put it, but preserves
        % self-contained nature of code.
        %
        function R = rotationR(~,th)
            R = [ cos(th), -sin(th) ;
                  sin(th),  cos(th) ];
        end
        
        %========================== linearize ==========================
        %
        % Returns the linearized system equations for use by control
        % synthesis or model reference schemes.
        %
        function [A,B] = linearize(self, x0, u0)
          % TO BE FILLED OUT IF WORTH DOING.
        end
        
        
    end
end

