classdef birotor < handle
    properties
        param = struct('m',     3,      ...
                       'Ixx',   0.05,   ...
                       'r',     0.175,  ...
                       'd',     0.03)   ;
        
        tSim
        xSim 
        xdotSim
        uSim
        
        % Most likely useful for other implementations, possibly as
        % subclass. Think through how coding design permits the different
        % versions to easily run.
        %A
        %B
        %P
        %Klqr
    end
    
    methods
        %=========================== birotor ===========================
        %
        % Constructor for the birotor instance.
        %
        function obj = birotor(param)
             if exist('param_in','var')
                  obj.param = param;
             end
        end
        
        %============================ runSim ===========================
        %
        % Abstracted interface to run the simulation of the ducted
        % fan. Uses a format similar to odeXX functions for specifying the
        % initial value problem (IVP) with the addition of a control
        % function.
        %
        function [tSim, xSim, xdotSim, uSim] = runSim(self, tspan, x0, u)
            sys = @(t,x) self.dynamics(t, x, u);

            % Simulate the system.
            [tSim, xSim] = ode45(sys, tspan, x0);

            % Use simulation data to recover controls and diff eq.
            % Data is row-wise.
            num     = length(tSim);
            uSim    = zeros(num,2);
            xdotSim = zeros(size(xSim));
            
            for i = 1:num
                x = xSim(i,:)';
                t = tSim(i);
                uSim(i,:) = u(t, x)';
                xdotSim(i,:) = self.dynamics(t, x, u)';
            end
            
            % Store variables in case needed for later.
            self.tSim = tSim;
            self.xSim = xSim;
            self.xdotSim = xdotSim;
            self.uSim = uSim;
        end
        
        %=========================== dynamics ==========================
        %
        % Compute the dynamics of the system under the applied control.
        %
        function dx = dynamics(self, t, xvec, uvec)
            % angular coordinate
            phi = xvec(3);
            
            % robot params
            m   = self.param.m;
            roI = self.param.r/self.param.Ixx;
            dom = self.param.d/self.param.m;
             
            % environment param
            g = 9.81;

            
            % transalational dynamics
            A = diag([1 1 1],3);
            A(4:5,4:5) = -dom*eye(2);

            B = [-sin(phi)/m,   0;                      ...
                  cos(phi)/m,   0;                      ...
                  0         ,  roI] * [1 1; 1 -1];


            % applied control.
            ut = uvec(t,xvec);

            
            % compute dynamics and stack derivatives
            yzphiddot = [0;-g;0] + B*ut;
            dx = A*xvec + [zeros(3,1) ; yzphiddot];
        end
        
        
        %======================= linearController ======================
        %
        % Implements linear error feedback law with acceleration based
        % feed-forward term if argument provided.
        %
        % Assumes that xref packaged to be function of time. 
        % Assumes that aref packaged to be function of time. 
        %
        function u = linearController(self, t, xvec, xref, aref)
            
            % Compute any feedforward component.
            if (nargin > 5)
              % Handle feedforward, otherwise there is none.
            else
              uFF = 0; % Make proper dimension.
            end

            % Compute error-feedback.

            % Pack into control.
            
        end

        %========================== linearize ==========================
        %
        % Returns the linearized system equations for use by control
        % synthesis or model reference schemes.
        %
        function [A,B] = linearize(self, x0, u0)
          % TO BE FILLED OUT IF WORTH DOING.
        end
        
                
    end
end

