classdef birotorAnimator < handle
    
    properties
        X
        U
        lastPlot
        vecMovie
        param
        fig = NaN;
        P;
    end
    
    methods
        function obj = birotorAnimator(fig)
             if exist('fig','var')
                  obj.fig = fig;
             end
             obj.P = [];
        end


        function newDrawing(self)
            self.P = [];
        end

        function isDrawn = haveDrawing(self)

            if (isempty(self.P))
                isDrawn = false;
            elseif all(isgraphics(self.P))
                isDrawn = true;
            else
                isDrawn = false;
            end

        end

        function animate(self, t, X, factor)
            if ~exist('factor','var')
                factor = 1;
            end
            if(factor<1)
                factor = 1;
                warning('factor has been clamped to 1. factor must be (factor>=1)!')
            end
            if ~isgraphics(self.fig)
                self.fig = figure();
            end
          
            % Use simulation limits to establish plot window size.
            % Keep square due to axis equal.
            wrad = 2;
            winy = [min(X(:,1))-wrad, max(X(:,1))+wrad];
            winz = [min(X(:,2))-wrad, max(X(:,2))+wrad];
                        
            % Check if have drawing from previous calls.
            if (self.haveDrawing)
              P = self.P;
            else
              P = [];
            end
            
            % Setup plot times.
            N   = size(X,1);
            tmp = 1:factor:N;
            if tmp(end) ~= N
                tmp = [tmp, N];
            end
            
            % Configure axes.
            axis equal;
            xlim(winy);
            ylim(winz);

            % Iterate through plot times and draw birotor.
            loop = 1;
            for i = tmp
                P = self.snapshot(X(i,:)', P);
                title("Time = " + sprintf('%.2f',t(i)) + " sec");
                drawnow;

                vec(loop) = getframe(self.fig);
                loop = loop+1;
            end
            self.vecMovie = vec;
            self.P = P;
        end
        
        function animateInput(self, t, X, U, factor)
            if ~exist('factor','var')
                factor = 1;
            end
            if(factor<1)
                factor = 1;
                warning('factor has been clamped to 1. factor must be (factor>=1)!')
            end
            if ~isgraphics(self.fig)
                self.fig = figure();
                self.fig.Position = [100 100 1000 400];
            end

            % Use simulation limits to establish plot window size.
            % Keep square due to axis equal.
            wrad = 2;
            winy = [min(X(:,1))-wrad, max(X(:,1))+wrad];
            winz = [min(X(:,2))-wrad, max(X(:,2))+wrad];

            % Get max and min over both control signals.
            % Add buffer above and below.
            urad = 1;
            y_ulim = [min(U(:))-urad, max(U(:))+urad];
            
            % Check if have drawing from previous calls.
            if (self.haveDrawing)
              P = self.P;
            else
              P = [];
            end

            Pu = [];

            % Setup plot times.
            N = size(X,1);
            tmp = 1:factor:N;
            if tmp(end) ~= N
                tmp = [tmp, N];
            end

            % Iterate through plot times and draw birotor.
            loop = 1;
            for i = tmp
                
                % Drawing of the birotor.
                h = subplot(1,2,1);
                if (isempty(P))
                  axis equal
                  xlim(winy)
                  ylim(winz)
                end
                P = self.snapshot(X(i,:)', P);
                title("Time = " + sprintf('%.2f',t(i)) + " sec")
                
                % Plot of evolving control signal.
                subplot(1,2,2)
                if (isempty(Pu))
                  Pu = plot(t(1:i), U(1:i,:)');
                  xlabel('Time (sec)')
                  ylabel('$u(t)$')
                  xlim([t(1), t(end)])
                  ylim(y_ulim)
                else
                  set(Pu(1), 'XData', t(1:i)', 'YData', U(1:i,1)');
                  set(Pu(2), 'XData', t(1:i)', 'YData', U(1:i,2)');
                end

                drawnow
                vec(loop) = getframe(self.fig);
                loop = loop+1;
            end
            self.P = P;
            self.vecMovie = vec;
        end

        function P = snapshot(self, X, P)
            if (nargin < 3)
                P = self.P;
            end
            
            tmp = abs(X(1,1)-X(end,1));
            tmpC = 10;
            if tmp < 1
               tmp = 1; 
            end

            axSize = 0.25;
            L = 0.5;
            
            y = X(1);
            z = X(2);
            phi = X(3);
            
            f1 = [y+L*cos(phi), z+L*sin(phi)];
            f2 = [y-L*cos(phi), z-L*sin(phi)];
            
            a1 = [y,z];
            a2 = [y+L*cos(pi/2+phi), z+L*sin(pi/2+phi)];
                        
            if (isempty(P))
                % axis (body z-axis)
                P(1) = line([a1(1), a2(1)], [a1(2), a2(2)], ...
                                                     'linewidth',3,'color','blue');
                % birotor frame (aligned with body y-axis)
                P(2) = line([f1(1), f2(1)], [f1(2), f2(2)], ...
                                                     'linewidth',3,'color','black');
            
                % birotor propellers
                washeld = ishold;
                hold on;
                p = nsidedpoly(10, 'Center', f1, 'Radius', 0.15);
                P(3) = plot(p, 'FaceColor', 'r');
                p = nsidedpoly(10, 'Center', f2, 'Radius', 0.15);
                P(4) = plot(p, 'FaceColor', 'r');
                if (~washeld)
                  hold off;
                end 
            else
                set(P(1), 'XData', [a1(1), a2(1)], 'YData', [a1(2), a2(2)]);
                set(P(2), 'XData', [f1(1), f2(1)], 'YData', [f1(2), f2(2)]);

                p = nsidedpoly(10, 'Center', f1, 'Radius', 0.15);
                set(P(3), 'Shape', p);
                p = nsidedpoly(10, 'Center', f2, 'Radius', 0.15);
                set(P(4), 'Shape', p);
            end
        end
        
        
        function export_movie(self, txt, fr)
            num = length(self.vecMovie);
            
            for i = 1:(num+10)
                if i <= 10
                    vec(i) = self.vecMovie(1);
                else
                    vec(i) = self.vecMovie(i-10);
                end
            end
%             myWriter = VideoWriter(txt, 'MPEG-4');
            myWriter = VideoWriter(txt);
            myWriter.FrameRate = fr;
            
            open(myWriter);
            writeVideo(myWriter, vec);
            close(myWriter)
        end
    end
end

