%================================ genMesh ================================
%
% @brief    Function that takes a point cloud, which should represent a
%           single surface object, and generates a triangulation for
%           plotting as a mesh.  
%
%  The triangulation algorithm is a naive version of the advancing fronts
%  (or surface tracking) algorithm.  Given a seed point, an initial
%  triangle is generated.  From this triangle, an expanding front of new
%  triangles is generated using existing edges from the previously
%  generated triangles. Given an edge, which define two of the triangle
%  points, a new triangle is found by providing a third point to form two
%  new edges.  This third point is either a pre-existing vertex or it is
%  an unused point in the space.  
% 
%  Each new triangle usually provides two new edges to process.  When
%  things start to get full, sometimes only one edge is missing, or
%  sometimes the triangle edges are already there, just that they are not
%  specified to be part of a triangle.  It gets tricky, but is fairly
%  straightforward.
%
%
%  @param[in]   thePts      The set of points to triangulate.
%  @param[in]   rho         Structure with proximity radii 
%                   .near   The nearby radius.
%                             Determines points in local neighborhood.
%                   .noise  The noise radius (something pretty small).
%                             Used to prevent round-off errors.
%                             Should be at level of expected error.
%
%  @param[out]  triVerts    The triangulation are list of vertices.
%  @param[out]  inPlay      List of points that are in play (for debugging).
%
%================================ genMesh ================================

%
% @ingroup  pointClouds
% @file     genMesh.m
%
% @author   Patricio A. Vela,       pvela@gatech.edu
% @date     2017/02/26
%
% @quitf
%================================ genMesh ================================
function [triVerts, inPlay] = genMesh(thePts, rho)


%-- Initialize main variables.
%
rawPoints = transpose(thePts.pts.Location);

inPlay    = true(1, thePts.numPoints());
edgeExist = sparse(thePts.numPoints(),thePts.numPoints());
isVertex  = false(1, thePts.numPoints());

triVerts  = [];
triEdges  = [];
edgePlanes = []; 

toProcess = [];

seedProcess(1);

%-- Run processingloop until no more vertices to add.
%
loops = 0;
maxLoop = 0;        %TODO: Change me. First start with 0 to make sure
                    %TODO:   that seed process is correct.
                    %TODO: Then change to 1 to test code.  Should add a
                    %TODO:   new triangle.
                    %TODO: Then slowly increment checking to see that new
                    %TODO:   triangles are proper.
                    %TODO: Once confident, just set equal to Inf to loop
                    %TODO:   until proper termination condition holds.

while (~isempty(toProcess) && loops < maxLoop)

  %-- Get the current edge pair to process and its normal plane equation.
  %
  eInds  = triEdges(:,toProcess(1));
  nPlane = edgePlanes(:,toProcess(1));

  %-- Get the center point of the edge.
  %
  centPt = pts3D(mean(rawPoints(:,eInds),2));

  %-- Find all points within circular radius of edge center point.
  %   Exclude the two vertices of the chosen edge.
  %   Get the actual point coordinates.
  %
  [npts, ninds] = centPt.findAdjacent(thePts, rho.near);
  %TODO: You need to write the findAdjacent function.
  %TODO: You should probably test it out on some baby data set
  %TODO:  that you make up.  Otherwise, you'll be in for a world of
  %TODO:  pain if this function does not work.

  exclude = find( (ninds == eInds(1)) | (ninds == eInds(2)) );
  npts.removeInds(exclude);
  ninds(exclude) = [];

  testPts = transpose(npts.pts.Location);
  testPts(end+1,:) = 1;

  %-- Compute the signed distance to the plane and get those in positive zone.
  %   If there are none, then, nothing to process, go to next point.
  %   If there are some, then extract those only, ignore the others.
  %
  planeDist = (nPlane' * testPts) - rho.noise; % Distance to plane of edge.
  posHalfPlane = find(planeDist > 0);          % Indices to "outside" points.

  if (isempty(posHalfPlane))
    toProcess(1) = [];
    disp('Nothing to process, must point outwards or nearby points taken.');
    continue;
  end

  negHalfPlane = find(planeDist <= 0);  % The inside point we don't want.
  ninds(negHalfPlane) = [];             % Get rid of them.
  planeDist(negHalfPlane) = [];

  nvinds = find(isVertex(ninds));       % Find subset that are vertices.
  edgeFound = false;

  %-- Check if non-existent triangle, but with existing edges can be formed
  %
  if (~isempty(nvinds)) 
    for di = nvinds
      eChecked  = checkEdges(eInds, ninds(di));
      edgeFound = all(eChecked > 0);

      if (edgeFound)
        pickInd = di;
        disp('Triangle found from existing edges!');
        break;
      end
    end
  end

  %-- Check if nearest vertex should be used to create an edge.
  %
  if (~isempty(nvinds) && ~edgeFound)
    disp('==== Otherwise, try connecting to nearest vertex.');

    [dvals, dinds] = sort(planeDist(nvinds),'ascend');
    for di = dinds
      cind = ninds(nvinds(di));
      [eChecked] = checkEdges(eInds, cind);
      if ( all(eChecked ~= 2) && any(eChecked == 0) ) 
        maxDist = planeDist(nvinds(di));
        pickInd  = nvinds(di);

        edgeFound = true;
        if (edgeFound)
          disp('Found existing vertex that can be connected!');
          break;
        end
      end
    end
  end

  if (~edgeFound)
    disp('===== No existing connections possible, pick a neighbor.');
    outSector = inPlay(ninds);
    ninds     = ninds(outSector);
    planeDist = planeDist(outSector);
    posHalfPlane = posHalfPlane(outSector);

    [maxDist, pickInd] = max(planeDist);
    edgeFound = ~isempty(pickInd);
  end

  if (~edgeFound)           % Nothing whatsoever found. Must be at edge.
    toProcess(1) = [];      % Move on.
    continue;
  end


  %-- Test is have already processed enough that going further would
  %   replicate efforts.
  tCheck = checkEdgesTriangle([eInds', ninds(pickInd)]);
  if (all(tCheck) > 1)
    toProcess(1) = [];
    disp('===== Appears to be a repeated triangle face.');
    continue;
  end

  [eCheck, sInds] = checkEdge(eInds);
  if (eCheck > 1)
    toProcess(1) = [];
    disp('===== Already processed this edge twice.');
    continue;
  else
    edgeExist(sInds(1), sInds(2)) = edgeExist(sInds(1), sInds(2)) + 1;
  end


  %--- Passed all sanity tests. Should instantiate as a triangle.
  %
  vertInd = ninds(pickInd);     % This is the third vertex to use.

  triangle = [eInds', vertInd] % The full triangle specification.

  [triangle, indperm] = sort(triangle, 'ascend');
  [indsort , indlut ] = sort(indperm);
  edgeInds = indlut([1 2; 3 3])

  pvects = [addTriangle(triangle, edgeInds)];
  excludeInterior(pvects, setdiff(ninds, triangle));

  [eCheck, sInds] = checkEdge([eInds(1), vertInd]);
  if (eCheck == 0)
    toProcess = [toProcess, size(triEdges,2)-1];
    edgeExist(sInds(1), sInds(2)) = 1;
  elseif (eCheck == 1)
    edgeExist(sInds(1), sInds(2)) = edgeExist(sInds(1), sInds(2)) + 1;
    inPlay(sInds) = false;
  end

  [eCheck, sInds] = checkEdge([eInds(2), vertInd]);
  if (eCheck == 0)
    toProcess = [toProcess, size(triEdges,2)];
    edgeExist(sInds(1), sInds(2)) = 1;
  elseif (eCheck == 1)
    edgeExist(sInds(1), sInds(2)) = edgeExist(sInds(1), sInds(2)) + 1;
    inPlay(sInds) = false;
  end

  toProcess(1) = [];
  loops = loops + 1;
end


%
%======================= Internal Helper Functions =======================
%
% The functions below have global access to variables in code above.
% They modify the main variables initialize at the start of the function.
%



  %============================ seedProcess ============================
  %
  %
  function seedProcess(seedInd)

  %--[1] Get the seed point and nearby points.
  %
  seedPoint = pts3D(rawPoints(:,seedInd));
  [npts, ninds] = seedPoint.findAdjacent(thePts, rho.near);
  ninds(ninds == seedInd) = [];

  %--[2] Use the first neighbor as an edge.
  %
  vect1 = -diff(rawPoints(:,[1,ninds(1)]), 1, 2);
  edgeV(:,1) = vect1/norm(vect1);

  %--[3] Compare all other neighbors and get one as close to orthogonal
  %      as possible, by minimizing the dot product of vectors to these
  %      points from the seed point.  End result is to pick smaller
  %      viable triangle at seed point.  Should pick bigger triangles
  %      during main loop, so no worries.
  %
  maxV = inf;
  maxI = 0;
  for ii = ninds(2:end)
    vect2 = -diff(rawPoints(:,[1,ii]), 1, 2);
    dp = abs(dot(edgeV(:,1),vect2));
    if (dp < maxV)
      edgeV(:,2) = vect2/norm(vect2);
      maxV  = dp;
      maxI  = ii;
    end
  end

  %--[4] Use the triplet as the seed triangle.
  %
  [triangle] = sort([1, ninds(1), maxI], 'ascend');
  edgeInds = [1 1 2; 2 3 3];   % edges: 1 -> 2, 1-> 3, and 2 -> 3.

  pvects = addTriangle(triangle, edgeInds);
  toProcess = [1, 2, 3];

  %--[5] All other nearby indices tested for being interior to triangle
  %      up to some uncertainty slack.  Will get those slightly outside
  %      based on the specified noise radius.
  %

  if (0)    %TODO: MAKE SURE TO REMOVE THE IF STATEMENT SO BELOW IS CALLED.
  excludeInterior(pvects, setdiff(ninds, triangle));
  end

  end


  %============================ addTriangle ============================
  %
  %
  % @param[in]  verts   The list of vertex points to use in the triangle.
  % @param[in]  toadd   The edges to add. Each column is edge specification.
  %                     There should be at least two new edges.
  %
  % @param[out] planeVects  The homogenous plane vector specifications.
  %                         As many as toadd has columns.
  %
  function planeVects = addTriangle(verts, toadd)

  %----[1] Append vertices to list and mark them as vertices.
  %
  triVerts = [triVerts; verts];
  isVertex(verts) = true;

  %----[2] Add edges to list of edges.
  %
  triEdges = [triEdges , verts(toadd)];

  %----[3] Compute the homogeneous plane vector.
  %        Add to list.
  %
  triCent  = mean( rawPoints(:,verts), 2 );

  for ti = 1:size(toadd,2)
    edge(:,ti) = -diff(rawPoints(:, verts(toadd(:,ti))), 1, 2);
    edge(:,ti) = edge(:,ti)/norm(edge(:,ti));
  end

  tNorm = cross(edge(:,1), edge(:,2));
  tNorm = tNorm/norm(tNorm);

  planeVects = [];
  for ti = 1:size(toadd,2)
    nP = [eye(3); -rawPoints(:,verts(toadd(1,ti)))']*cross(edge(:,ti), tNorm);
    if ( dot(nP,[triCent; 1]) > 0 )
      nP = -nP;
    end
    planeVects = [planeVects, nP];
  end

  edgePlanes = [edgePlanes, planeVects];

  end

  %========================== excludeInterior ==========================
  %
  % @brief  Given a set of homogeneous plane vectors, test a specified
  %         set of points for negative distance with respect to all of
  %         the homogeneous plane vectors given.
  %
  %  To check a single homogeneous plane vector involves 
  %
  %     transpose(nPlane) * testPt <= rho_noise
  %
  %  A matrix of plane vectors would behace similarly, but return
  %  multiple rows, one for each plane vector.  To get the testPt, we
  %  assume that the indices passed can be used with the rawPoints
  %  matrix, which has the points columnwise. Of course these points have
  %  to be put in homogeneous form to work out.
  %
  % @param[in]  pvects  The homogeneous plane vectors as a matrix.
  %                     Each column is a honogeneous plane vector.
  %
  % @param[in]  ninds   The indices of the points to check.
  %
  function excludeInterior(pvects, ninds)

  if (isempty(ninds))
    return;
  end

  %-- Check all specified points for interiorness to passed planes.
  inTest =  PLANE_EQUATION_HERE <= rho.noise;
  inTest = logical(prod(inTest,1));

  %-- Those that pass the interior test are no longer useful (not in play).
  interiorInds = ninds(inTest);
  inPlay(WHATWHAT) = false;

  end

  %============================== checkEdge ============================
  %
  % @brief  Sorts the edge pair and queries edgeExist to get the
  %         edge existence value.
  %
  %  The edge existence values are: 0 - not used, 1 - one edge, and 2 -
  %  two edges.  An edge should have just two triangles associated to it.
  %  The edgeExists variable only has the upper right half populated
  %  (since lower half would mean the same things).  The means look-up
  %  requires first index to be smaller than second index.
  %
  % @param[in]  einds   The two vertex indices to check for edge value.
  %
  % @param[out] checkOut    The edgeExists values checked.
  % @param[out] sortInds    The sorted indices in case needed.
  %
  function [checkOut, sortInds] = checkEdge(einds)

  sortInds =  FILLMEIN;
  checkOut  = edgeExist( FILLMEIN );

  end

  %============================= checkEdges ============================
  %
  % @brief  Compare a few vertics against a given vertex index.
  %
  %  Rather than check a bunch of edges with the same final vertex, we
  %  send here all of the start vertices (as indices) as an argument and
  %  the final vertex as a second argument. A loop will check each pair
  %  to start to final vertices.
  %
  % @param[in]  inds    The start vertices to check.
  % @param[in]  tind    This is the second vertex to each start vertex.
  %
  % @param[out] checkOut    The results for each start-final pair.
  %
  function checkOut = checkEdges(inds, tind)

  checkOut = zeros(1, numel(inds));

  for ii=1:numel(inds)
    checkInds = FILLMEIN_LIKE_IN_CHECKEDGE;
    checkOut(ii) = edgeExist( FILLMEIN );
  end

  end

  %========================= checkEdgesTriangle ========================
  %
  % @brief  Evaluate a vertex triplet as though it were a triangle.
  %         Return the edge existence values.
  %
  % @param[in]  tinds   A triplet of indices, supposedly for a triangle.
  %
  function checkOut = checkEdgesTriangle(tinds)

  checkOut = zeros(1, 3);

  checkInds = SORTME ;
  checkOut(1) = edgeExist(checkInds(1),checkInds(2));
  checkOut(2) = edgeExist(checkInds(1),checkInds(3));
  checkOut(3) = edgeExist(checkInds(2),checkInds(3));

  end

end
